VERSION 1.0 CLASS
BEGIN
  MultiUse = -1  'True
END
Attribute VB_Name = "LesserScraping"
Attribute VB_GlobalNameSpace = False
Attribute VB_Creatable = False
Attribute VB_PredeclaredId = True
Attribute VB_Exposed = True
'===============================================
' LesserScraping - VBA Web Scraping Class
' Version: 2.1.0
' Author: funcref.com
' License: MIT
' https://opensource.org/licenses/mit-license.php
'===============================================
Option Explicit

'-----------------------------------------------
' Windows API Declarations
'-----------------------------------------------
#If VBA7 Then
    Private Declare PtrSafe Function URLDownloadToFile Lib "urlmon" Alias "URLDownloadToFileA" _
        (ByVal pCaller As LongPtr, ByVal szURL As String, ByVal szFileName As String, _
         ByVal dwReserved As Long, ByVal lpfnCB As LongPtr) As Long
    Private Declare PtrSafe Function DeleteUrlCacheEntry Lib "wininet" Alias "DeleteUrlCacheEntryA" _
        (ByVal lpszUrlName As String) As Long
#Else
    Private Declare Function URLDownloadToFile Lib "urlmon" Alias "URLDownloadToFileA" _
        (ByVal pCaller As Long, ByVal szURL As String, ByVal szFileName As String, _
         ByVal dwReserved As Long, ByVal lpfnCB As Long) As Long
    Private Declare Function DeleteUrlCacheEntry Lib "wininet" Alias "DeleteUrlCacheEntryA" _
        (ByVal lpszUrlName As String) As Long
#End If

'-----------------------------------------------
' Enums
'-----------------------------------------------
Public Enum LocatorType
    ById = 1
    ByName = 2
    ByClassName = 3
    ByXPath = 4
    ByCssSelector = 5
    ByLinkText = 6
    ByPartialLinkText = 7
    ByTagName = 8
End Enum

Public Enum SpecialKey
    KeyBackspace = 1
    KeyTab = 2
    KeyEnter = 3
    KeyEscape = 4
    KeySpace = 5
    KeyDelete = 6
    KeyArrowUp = 7
    KeyArrowDown = 8
    KeyArrowLeft = 9
    KeyArrowRight = 10
End Enum

'-----------------------------------------------
' Constants
'-----------------------------------------------
Private Const ELEMENT_ID_KEY As String = "element-6066-11e4-a52e-4f735466cecf"
Private Const BASE_FOLDER As String = "LesserScraping"
Private Const DRIVER_FOLDER As String = "\driver"
Private Const BROWSER_FOLDER As String = "\browser"
Private Const PROFILE_FOLDER As String = "\profile"
Private Const DEFAULT_TIMEOUT_MS As Long = 10000
Private Const DEFAULT_POLL_INTERVAL_MS As Long = 100

'-----------------------------------------------
' Module Level Variables
'-----------------------------------------------
Private m_BrowserVersion As String
Private m_DriverVersion As String
Private m_BrowserPort As String
Private m_DriverPort As String
Private m_ProfileName As String
Private m_SessionId As String
Private m_CurrentElementId As String
Private m_Headless As Boolean
Private m_ImplicitWaitMs As Long
Private m_LastError As String

'-----------------------------------------------
' Cached Objects (Singleton Pattern)
'-----------------------------------------------
Private m_FSO As Object
Private m_WSH As Object

'===============================================
' Class Events
'===============================================
Private Sub Class_Initialize()
    m_BrowserVersion = DetectChromeVersion()
    If Not IsValidVersionFormat(m_BrowserVersion) Then
        RaiseError "Google Chrome ܂BCXg[Ă邩mFĂB"
    End If
    
    m_ImplicitWaitMs = DEFAULT_TIMEOUT_MS
    
    EnsureFolderExists BasePath & DRIVER_FOLDER
    EnsureFolderExists BasePath & BROWSER_FOLDER
    EnsureFolderExists BasePath & PROFILE_FOLDER
End Sub

Private Sub Class_Terminate()
    On Error Resume Next
    If m_Headless Then
        Quit True
    Else
        Quit False
    End If
    On Error GoTo 0
End Sub

'===============================================
' Public Properties
'===============================================
Public Property Get ChromeBrowserVersion() As String
    ChromeBrowserVersion = m_BrowserVersion
End Property

Public Property Get ChromeDriverVersion() As String
    ChromeDriverVersion = m_DriverVersion
End Property

Public Property Let ChromeDriverVersion(ByVal Value As String)
    If Not IsValidVersionFormat(Value) Then
        RaiseError "ȃhCo[o[W`ł: " & Value
    End If
    m_DriverVersion = Value
End Property

Public Property Let ChromeProfileName(ByVal Value As String)
    m_ProfileName = SanitizeFileName(Value)
End Property

Public Property Let ChromeUsedPort(ByVal Value As Long)
    If Value < 1024 Or Value > 65535 Then
        RaiseError "|[gԍ1024`65535͈̔͂Ŏw肵Ă: " & Value
    End If
    m_BrowserPort = CStr(Value)
End Property

Public Property Let ChromeDriverPort(ByVal Value As Long)
    If Value < 1024 Or Value > 65535 Then
        RaiseError "|[gԍ1024`65535͈̔͂Ŏw肵Ă: " & Value
    End If
    m_DriverPort = CStr(Value)
End Property

Public Property Let Headless(ByVal Value As Boolean)
    m_Headless = Value
End Property

Public Property Get Headless() As Boolean
    Headless = m_Headless
End Property

Public Property Let ImplicitWait(ByVal Milliseconds As Long)
    m_ImplicitWaitMs = Milliseconds
End Property

Public Property Get ImplicitWait() As Long
    ImplicitWait = m_ImplicitWaitMs
End Property

Public Property Get LastError() As String
    LastError = m_LastError
End Property

Public Property Get sessionId() As String
    sessionId = m_SessionId
End Property

'===============================================
' Public Methods - Session Control
'===============================================
Public Sub Start()
    If m_ProfileName = "" Then m_ProfileName = "Temp_chrome"
    If m_BrowserPort = "" Then m_BrowserPort = "9223"
    If m_DriverVersion = "" Then m_DriverVersion = GetMatchingDriverVersion(m_BrowserVersion)
    If m_DriverPort = "" Then m_DriverPort = "9515"
    
    EnsureDriverExists
    
    On Error Resume Next
    wsh.Run "taskkill /F /IM chromedriver_" & m_DriverVersion & ".exe", 0, False
    On Error GoTo 0
    
    Wait 200
    
    shell driverPath & " --port=" & m_DriverPort, vbHide
    
    If Not WaitForDriverReady(5000) Then
        RaiseError "ChromeDriver̋NɎs܂"
    End If
    
    CreateBrowserShortcut
    OpenBrowserIfNotRunning
    
    Dim retryCount As Long
    Do
        retryCount = retryCount + 1
        DoEvents
        Wait 200
        If retryCount > 25 Then
            RaiseError "uEUւ̐ڑɎs܂B" & vbCrLf & _
                       "Chrome NĂ邩mFĂB" & vbCrLf & _
                       "|[g " & m_BrowserPort & " ̃AvP[VŎgpĂ\܂B"
        End If
    Loop Until ConnectToBrowser()
End Sub

Private Function WaitForDriverReady(ByVal TimeoutMs As Long) As Boolean
    Dim startTime As Double: startTime = Timer
    Dim elapsed As Double
    Dim http As Object
    
    Do
        On Error Resume Next
        Set http = CreateObject("MSXML2.ServerXMLHTTP")
        http.setTimeouts 1000, 1000, 1000, 1000
        http.Open "GET", DriverBaseUrl & "status", False
        http.send
        
        If http.status = 200 Then
            WaitForDriverReady = True
            Exit Function
        End If
        On Error GoTo 0
        
        Wait 100
        elapsed = (Timer - startTime) * 1000
        If startTime > Timer Then elapsed = elapsed + 86400000
    Loop While elapsed < TimeoutMs
    
    WaitForDriverReady = False
End Function

Public Sub Quit(Optional ByVal CloseAllWindows As Boolean = False)
    On Error Resume Next
    
    If CloseAllWindows And m_SessionId <> "" Then
        Dim handles As Collection
        Set handles = getAllWindowHandles()
        If Not handles Is Nothing Then
            Dim i As Long
            For i = 1 To handles.count
                CloseCurrentWindow
            Next i
        End If
    End If
    
    If m_SessionId <> "" Then
        SendRequestInternal "DELETE", SessionUrl, Nothing, True
    End If
    
    SendRequestInternal "POST", DriverBaseUrl & "shutdown", Nothing, True
    
    If m_Headless Then
        Wait 300
        wsh.Run "cmd /c FOR /F ""tokens=5"" %a IN ('netstat -ano ^| findstr :" & m_BrowserPort & "') DO taskkill /F /PID %a", 0, True
    End If
    
    m_SessionId = ""
    
    On Error GoTo 0
End Sub

Public Sub Wait(ByVal Milliseconds As Long)
    Dim endTime As Double
    endTime = Timer + Milliseconds / 1000
    Do While Timer < endTime
        DoEvents
    Loop
End Sub

'===============================================
' Public Methods - Navigation
'===============================================
Public Sub NavigateTo(ByVal url As String, Optional ByVal WaitMs As Long = 100)
    Dim params As Object
    Set params = CreateObject("Scripting.Dictionary")
    params.Add "url", url
    sendRequest "POST", SessionUrl & "/url", params
    Wait WaitMs
End Sub

Public Function GetPageTitle() As String
    GetPageTitle = sendRequest("GET", SessionUrl & "/title", Nothing)("value")
End Function

Public Function GetCurrentUrl() As String
    GetCurrentUrl = sendRequest("GET", SessionUrl & "/url", Nothing)("value")
End Function

Public Function GetPageSource() As String
    GetPageSource = sendRequest("GET", SessionUrl & "/source", Nothing)("value")
End Function

Public Sub Refresh()
    sendRequest "POST", SessionUrl & "/refresh", CreateObject("Scripting.Dictionary")
End Sub

Public Sub GoBack()
    sendRequest "POST", SessionUrl & "/back", CreateObject("Scripting.Dictionary")
End Sub

Public Sub GoForward()
    sendRequest "POST", SessionUrl & "/forward", CreateObject("Scripting.Dictionary")
End Sub

'===============================================
' Public Methods - Wait & Synchronization
'===============================================
Public Function WaitForElement(ByVal Locator As String, ByVal LocType As LocatorType, _
                               Optional ByVal TimeoutMs As Long = -1, _
                               Optional ByVal index As Long = 1) As Boolean
    If TimeoutMs < 0 Then TimeoutMs = m_ImplicitWaitMs
    
    Dim startTime As Double: startTime = Timer
    Dim elapsed As Double
    
    Do
        If ExistElement(Locator, LocType, index) Then
            WaitForElement = True
            Exit Function
        End If
        Wait DEFAULT_POLL_INTERVAL_MS
        elapsed = (Timer - startTime) * 1000
        If startTime > Timer Then elapsed = elapsed + 86400000
    Loop While elapsed < TimeoutMs
    
    m_LastError = "vf܂ł: " & Locator & " (^CAEg: " & TimeoutMs & "ms)"
    WaitForElement = False
End Function

Public Function WaitForElementVisible(ByVal Locator As String, ByVal LocType As LocatorType, _
                                      Optional ByVal TimeoutMs As Long = -1, _
                                      Optional ByVal index As Long = 1) As Boolean
    If TimeoutMs < 0 Then TimeoutMs = m_ImplicitWaitMs
    
    Dim startTime As Double: startTime = Timer
    Dim elapsed As Double
    
    Do
        If ExistElement(Locator, LocType, index) Then
            If IsElementDisplayed(Locator, LocType, index) Then
                WaitForElementVisible = True
                Exit Function
            End If
        End If
        Wait DEFAULT_POLL_INTERVAL_MS
        elapsed = (Timer - startTime) * 1000
        If startTime > Timer Then elapsed = elapsed + 86400000
    Loop While elapsed < TimeoutMs
    
    m_LastError = "\\ȗvf܂ł: " & Locator
    WaitForElementVisible = False
End Function

Public Function WaitForElementClickable(ByVal Locator As String, ByVal LocType As LocatorType, _
                                        Optional ByVal TimeoutMs As Long = -1, _
                                        Optional ByVal index As Long = 1) As Boolean
    If TimeoutMs < 0 Then TimeoutMs = m_ImplicitWaitMs
    
    Dim startTime As Double: startTime = Timer
    Dim elapsed As Double
    
    Do
        If ExistElement(Locator, LocType, index) Then
            If IsElementDisplayed(Locator, LocType, index) And IsElementEnabled(Locator, LocType, index) Then
                WaitForElementClickable = True
                Exit Function
            End If
        End If
        Wait DEFAULT_POLL_INTERVAL_MS
        elapsed = (Timer - startTime) * 1000
        If startTime > Timer Then elapsed = elapsed + 86400000
    Loop While elapsed < TimeoutMs
    
    m_LastError = "NbN\ȗvf܂ł: " & Locator
    WaitForElementClickable = False
End Function

Public Function WaitForPageLoad(Optional ByVal TimeoutMs As Long = -1) As Boolean
    If TimeoutMs < 0 Then TimeoutMs = m_ImplicitWaitMs
    
    Dim startTime As Double: startTime = Timer
    Dim elapsed As Double
    Dim result As Object
    
    Do
        Set result = ExecuteScript("return document.readyState;", , 0)
        If Not result Is Nothing Then
            If result("value") = "complete" Then
                WaitForPageLoad = True
                Exit Function
            End If
        End If
        Wait DEFAULT_POLL_INTERVAL_MS
        elapsed = (Timer - startTime) * 1000
        If startTime > Timer Then elapsed = elapsed + 86400000
    Loop While elapsed < TimeoutMs
    
    m_LastError = "y[W̓ǂݍ݂^CAEg܂"
    WaitForPageLoad = False
End Function

Public Function WaitForUrlContains(ByVal PartialUrl As String, _
                                   Optional ByVal TimeoutMs As Long = -1) As Boolean
    If TimeoutMs < 0 Then TimeoutMs = m_ImplicitWaitMs
    
    Dim startTime As Double: startTime = Timer
    Dim elapsed As Double
    
    Do
        If InStr(GetCurrentUrl(), PartialUrl) > 0 Then
            WaitForUrlContains = True
            Exit Function
        End If
        Wait DEFAULT_POLL_INTERVAL_MS
        elapsed = (Timer - startTime) * 1000
        If startTime > Timer Then elapsed = elapsed + 86400000
    Loop While elapsed < TimeoutMs
    
    m_LastError = "URL '" & PartialUrl & "' ܂܂܂ł"
    WaitForUrlContains = False
End Function

'===============================================
' Public Methods - JavaScript
'===============================================
Public Function ExecuteScript(ByVal script As String, _
                              Optional ByRef args As Variant, _
                              Optional ByVal WaitMs As Long = 100) As Object
    Dim params As Object
    Set params = CreateObject("Scripting.Dictionary")
    params.Add "script", script
    
    If IsMissing(args) Then
        params.Add "args", Array()
    ElseIf IsArray(args) Then
        params.Add "args", args
    Else
        params.Add "args", Array(args)
    End If
    
    Set ExecuteScript = sendRequest("POST", SessionUrl & "/execute/sync", params)
    If WaitMs > 0 Then Wait WaitMs
End Function

Public Function ExecuteAsyncScript(ByVal script As String, _
                                   Optional ByRef args As Variant, _
                                   Optional ByVal WaitMs As Long = 100) As Object
    Dim params As Object
    Set params = CreateObject("Scripting.Dictionary")
    params.Add "script", script
    
    If IsMissing(args) Then
        params.Add "args", Array()
    ElseIf IsArray(args) Then
        params.Add "args", args
    Else
        params.Add "args", Array(args)
    End If
    
    Set ExecuteAsyncScript = sendRequest("POST", SessionUrl & "/execute/async", params)
    If WaitMs > 0 Then Wait WaitMs
End Function

Public Sub ScrollTo(ByVal Locator As String, ByVal LocType As LocatorType, _
                    Optional ByVal index As Long = 1, Optional ByVal WaitMs As Long = 100)
    FindElement Locator, LocType, index
    ExecuteScript "arguments[0].scrollIntoView({behavior:'smooth',block:'center'});", _
                  Array(CreateElementReference(m_CurrentElementId)), 0
    Wait WaitMs
End Sub

Public Sub ScrollToTop()
    ExecuteScript "window.scrollTo(0,0);", , 100
End Sub

Public Sub ScrollToBottom()
    ExecuteScript "window.scrollTo(0,document.body.scrollHeight);", , 100
End Sub

Public Sub ScrollBy(ByVal x As Long, ByVal y As Long)
    ExecuteScript "window.scrollBy(" & x & "," & y & ");", , 100
End Sub

'===============================================
' Public Methods - Element Interaction
'===============================================
Public Function GetAttribute(ByVal AttrName As String, ByVal Locator As String, _
                             ByVal LocType As LocatorType, Optional ByVal index As Long = 1) As String
    FindElement Locator, LocType, index
    Dim result As Object
    Set result = sendRequest("GET", SessionUrl & "/element/" & m_CurrentElementId & "/attribute/" & AttrName, Nothing)
    If result.Exists("value") Then
        If Not IsNull(result("value")) Then
            GetAttribute = result("value")
        End If
    End If
End Function

Public Function GetProperty(ByVal PropName As String, ByVal Locator As String, _
                            ByVal LocType As LocatorType, Optional ByVal index As Long = 1) As String
    FindElement Locator, LocType, index
    Dim result As Object
    Set result = sendRequest("GET", SessionUrl & "/element/" & m_CurrentElementId & "/property/" & PropName, Nothing)
    If result.Exists("value") Then
        If Not IsNull(result("value")) Then
            GetProperty = result("value")
        End If
    End If
End Function

Public Function GetText(ByVal Locator As String, ByVal LocType As LocatorType, _
                        Optional ByVal index As Long = 1) As String
    FindElement Locator, LocType, index
    GetText = sendRequest("GET", SessionUrl & "/element/" & m_CurrentElementId & "/text", Nothing)("value")
End Function

Public Function GetValue(ByVal Locator As String, ByVal LocType As LocatorType, _
                         Optional ByVal index As Long = 1) As String
    GetValue = GetProperty("value", Locator, LocType, index)
End Function

Public Function GetCssValue(ByVal PropName As String, ByVal Locator As String, _
                            ByVal LocType As LocatorType, Optional ByVal index As Long = 1) As String
    FindElement Locator, LocType, index
    Dim result As Object
    Set result = sendRequest("GET", SessionUrl & "/element/" & m_CurrentElementId & "/css/" & PropName, Nothing)
    If result.Exists("value") Then
        GetCssValue = result("value")
    End If
End Function

Public Function GetTagName(ByVal Locator As String, ByVal LocType As LocatorType, _
                           Optional ByVal index As Long = 1) As String
    FindElement Locator, LocType, index
    GetTagName = sendRequest("GET", SessionUrl & "/element/" & m_CurrentElementId & "/name", Nothing)("value")
End Function

Public Function GetRect(ByVal Locator As String, ByVal LocType As LocatorType, _
                        Optional ByVal index As Long = 1) As Object
    FindElement Locator, LocType, index
    Set GetRect = sendRequest("GET", SessionUrl & "/element/" & m_CurrentElementId & "/rect", Nothing)("value")
End Function

Public Sub Click(ByVal Locator As String, ByVal LocType As LocatorType, _
                 Optional ByVal index As Long = 1, Optional ByVal WaitMs As Long = 100)
    FindElement Locator, LocType, index
    sendRequest "POST", SessionUrl & "/element/" & m_CurrentElementId & "/click", CreateObject("Scripting.Dictionary")
    Wait WaitMs
End Sub

Public Sub DoubleClick(ByVal Locator As String, ByVal LocType As LocatorType, _
                       Optional ByVal index As Long = 1, Optional ByVal WaitMs As Long = 100)
    FindElement Locator, LocType, index
    ExecuteScript "var e=arguments[0];var evt=new MouseEvent('dblclick',{bubbles:true,cancelable:true,view:window});e.dispatchEvent(evt);", _
                  Array(CreateElementReference(m_CurrentElementId)), 0
    Wait WaitMs
End Sub

Public Sub RightClick(ByVal Locator As String, ByVal LocType As LocatorType, _
                      Optional ByVal index As Long = 1, Optional ByVal WaitMs As Long = 100)
    FindElement Locator, LocType, index
    ExecuteScript "var e=arguments[0];var evt=new MouseEvent('contextmenu',{bubbles:true,cancelable:true,view:window});e.dispatchEvent(evt);", _
                  Array(CreateElementReference(m_CurrentElementId)), 0
    Wait WaitMs
End Sub

Public Sub Hover(ByVal Locator As String, ByVal LocType As LocatorType, _
                 Optional ByVal index As Long = 1, Optional ByVal WaitMs As Long = 100)
    FindElement Locator, LocType, index
    ExecuteScript "var e=arguments[0];var evt=new MouseEvent('mouseover',{bubbles:true,cancelable:true,view:window});e.dispatchEvent(evt);", _
                  Array(CreateElementReference(m_CurrentElementId)), 0
    Wait WaitMs
End Sub

Public Sub ClearText(ByVal Locator As String, ByVal LocType As LocatorType, _
                     Optional ByVal index As Long = 1)
    FindElement Locator, LocType, index
    sendRequest "POST", SessionUrl & "/element/" & m_CurrentElementId & "/clear", CreateObject("Scripting.Dictionary")
End Sub

Public Sub SetText(ByVal text As String, ByVal Locator As String, ByVal LocType As LocatorType, _
                   Optional ByVal index As Long = 1, Optional ByVal OverWrite As Boolean = True, _
                   Optional ByVal WaitMs As Long = 100)
    If OverWrite Then ClearText Locator, LocType, index
    
    Dim params As Object
    Set params = CreateObject("Scripting.Dictionary")
    params.Add "text", text
    params.Add "value", StringToCharArray(text)
    
    FindElement Locator, LocType, index
    sendRequest "POST", SessionUrl & "/element/" & m_CurrentElementId & "/value", params
    Wait WaitMs
End Sub

Public Sub SendKey(ByVal Key As SpecialKey, ByVal Locator As String, ByVal LocType As LocatorType, _
                   Optional ByVal index As Long = 1, Optional ByVal WaitMs As Long = 100)
    Dim params As Object
    Set params = CreateObject("Scripting.Dictionary")
    Dim keyChar As String
    
    Select Case Key
        Case KeyBackspace: keyChar = ChrW(&HE003)
        Case KeyTab:       keyChar = ChrW(&HE004)
        Case KeyEnter:     keyChar = ChrW(&HE007)
        Case KeyEscape:    keyChar = ChrW(&HE00C)
        Case KeySpace:     keyChar = ChrW(&HE00D)
        Case KeyDelete:    keyChar = ChrW(&HE017)
        Case KeyArrowUp:   keyChar = ChrW(&HE013)
        Case KeyArrowDown: keyChar = ChrW(&HE015)
        Case KeyArrowLeft: keyChar = ChrW(&HE012)
        Case KeyArrowRight: keyChar = ChrW(&HE014)
    End Select
    
    params.Add "text", keyChar
    
    FindElement Locator, LocType, index
    sendRequest "POST", SessionUrl & "/element/" & m_CurrentElementId & "/value", params
    Wait WaitMs
End Sub

Public Sub Focus(ByVal Locator As String, ByVal LocType As LocatorType, _
                 Optional ByVal index As Long = 1)
    FindElement Locator, LocType, index
    ExecuteScript "arguments[0].scrollIntoView({behavior:'auto',block:'center'});", _
                  Array(CreateElementReference(m_CurrentElementId)), 0
    On Error Resume Next
    sendRequest "POST", SessionUrl & "/element/" & m_CurrentElementId & "/click", _
                CreateObject("Scripting.Dictionary")
    On Error GoTo 0
    ExecuteScript "arguments[0].focus();", Array(CreateElementReference(m_CurrentElementId)), 0
End Sub

Public Sub Blur(ByVal Locator As String, ByVal LocType As LocatorType, _
                Optional ByVal index As Long = 1)
    FindElement Locator, LocType, index
    ExecuteScript "arguments[0].blur();", Array(CreateElementReference(m_CurrentElementId)), 0
End Sub

Public Function ExistElement(ByVal Locator As String, ByVal LocType As LocatorType, _
                             Optional ByVal index As Long = 1) As Boolean
    On Error Resume Next
    m_CurrentElementId = ""
    FindElement Locator, LocType, index
    ExistElement = (m_CurrentElementId <> "")
    On Error GoTo 0
End Function

Public Function IsElementDisplayed(ByVal Locator As String, ByVal LocType As LocatorType, _
                                   Optional ByVal index As Long = 1) As Boolean
    On Error Resume Next
    FindElement Locator, LocType, index
    If m_CurrentElementId = "" Then
        IsElementDisplayed = False
        Exit Function
    End If
    Dim result As Object
    Set result = sendRequest("GET", SessionUrl & "/element/" & m_CurrentElementId & "/displayed", Nothing)
    IsElementDisplayed = result("value")
    On Error GoTo 0
End Function

Public Function IsElementEnabled(ByVal Locator As String, ByVal LocType As LocatorType, _
                                 Optional ByVal index As Long = 1) As Boolean
    On Error Resume Next
    FindElement Locator, LocType, index
    If m_CurrentElementId = "" Then
        IsElementEnabled = False
        Exit Function
    End If
    Dim result As Object
    Set result = sendRequest("GET", SessionUrl & "/element/" & m_CurrentElementId & "/enabled", Nothing)
    IsElementEnabled = result("value")
    On Error GoTo 0
End Function

Public Function IsElementSelected(ByVal Locator As String, ByVal LocType As LocatorType, _
                                  Optional ByVal index As Long = 1) As Boolean
    On Error Resume Next
    FindElement Locator, LocType, index
    If m_CurrentElementId = "" Then
        IsElementSelected = False
        Exit Function
    End If
    Dim result As Object
    Set result = sendRequest("GET", SessionUrl & "/element/" & m_CurrentElementId & "/selected", Nothing)
    IsElementSelected = result("value")
    On Error GoTo 0
End Function

Public Function GetElementCount(ByVal Locator As String, ByVal LocType As LocatorType) As Long
    On Error Resume Next
    Dim elements As Collection
    Set elements = FindElements(Locator, LocType)
    If elements Is Nothing Then
        GetElementCount = 0
    Else
        GetElementCount = elements.count
    End If
    On Error GoTo 0
End Function

'===============================================
' Public Methods - Multiple Elements
'===============================================
Public Function GetTexts(ByVal Locator As String, ByVal LocType As LocatorType) As Collection
    Set GetTexts = New Collection
    Dim elements As Collection
    Set elements = FindElements(Locator, LocType)
    
    If elements Is Nothing Then Exit Function
    
    Dim elem As Object, text As String, result As Object
    Dim i As Long
    For i = 1 To elements.count
        Set elem = elements(i)
        Set result = sendRequest("GET", SessionUrl & "/element/" & elem(ELEMENT_ID_KEY) & "/text", Nothing)
        If result.Exists("value") Then
            GetTexts.Add result("value")
        Else
            GetTexts.Add ""
        End If
    Next i
End Function

Public Function GetAttributes(ByVal AttrName As String, ByVal Locator As String, _
                              ByVal LocType As LocatorType) As Collection
    Set GetAttributes = New Collection
    Dim elements As Collection
    Set elements = FindElements(Locator, LocType)
    
    If elements Is Nothing Then Exit Function
    
    Dim elem As Object, result As Object
    Dim i As Long
    For i = 1 To elements.count
        Set elem = elements(i)
        Set result = sendRequest("GET", SessionUrl & "/element/" & elem(ELEMENT_ID_KEY) & "/attribute/" & AttrName, Nothing)
        If result.Exists("value") And Not IsNull(result("value")) Then
            GetAttributes.Add result("value")
        Else
            GetAttributes.Add ""
        End If
    Next i
End Function

Public Function GetValues(ByVal Locator As String, ByVal LocType As LocatorType) As Collection
    Set GetValues = GetAttributes("value", Locator, LocType)
End Function

'===============================================
' Public Methods - Select (Dropdown)
'===============================================
Public Sub SelectByValue(ByVal Value As String, ByVal Locator As String, _
                         ByVal LocType As LocatorType, Optional ByVal index As Long = 1)
    FindElement Locator, LocType, index
    Dim script As String
    script = "var s=arguments[0];" & _
             "for(var i=0;i<s.options.length;i++){" & _
             "if(s.options[i].value==='" & EscapeJsString(Value) & "'){" & _
             "s.selectedIndex=i;" & _
             "s.dispatchEvent(new Event('change',{bubbles:true}));break;}}"
    ExecuteScript script, Array(CreateElementReference(m_CurrentElementId)), 0
End Sub

Public Sub SelectByText(ByVal text As String, ByVal Locator As String, _
                        ByVal LocType As LocatorType, Optional ByVal index As Long = 1)
    FindElement Locator, LocType, index
    Dim script As String
    script = "var s=arguments[0];" & _
             "for(var i=0;i<s.options.length;i++){" & _
             "if(s.options[i].text==='" & EscapeJsString(text) & "'){" & _
             "s.selectedIndex=i;" & _
             "s.dispatchEvent(new Event('change',{bubbles:true}));break;}}"
    ExecuteScript script, Array(CreateElementReference(m_CurrentElementId)), 0
End Sub

Public Sub SelectByIndex(ByVal OptionIndex As Long, ByVal Locator As String, _
                         ByVal LocType As LocatorType, Optional ByVal index As Long = 1)
    FindElement Locator, LocType, index
    Dim script As String
    script = "var s=arguments[0];" & _
             "if(" & OptionIndex & "<s.options.length){" & _
             "s.selectedIndex=" & OptionIndex & ";" & _
             "s.dispatchEvent(new Event('change',{bubbles:true}));}"
    ExecuteScript script, Array(CreateElementReference(m_CurrentElementId)), 0
End Sub

Public Function GetSelectedValue(ByVal Locator As String, ByVal LocType As LocatorType, _
                                 Optional ByVal index As Long = 1) As String
    FindElement Locator, LocType, index
    Dim result As Object
    Set result = ExecuteScript("return arguments[0].value;", Array(CreateElementReference(m_CurrentElementId)), 0)
    If Not IsNull(result("value")) Then
        GetSelectedValue = result("value")
    End If
End Function

Public Function GetSelectedText(ByVal Locator As String, ByVal LocType As LocatorType, _
                                Optional ByVal index As Long = 1) As String
    FindElement Locator, LocType, index
    Dim result As Object
    Set result = ExecuteScript("var s=arguments[0];return s.options[s.selectedIndex].text;", _
                               Array(CreateElementReference(m_CurrentElementId)), 0)
    If Not IsNull(result("value")) Then
        GetSelectedText = result("value")
    End If
End Function

Public Function GetSelectOptions(ByVal Locator As String, ByVal LocType As LocatorType, _
                                 Optional ByVal index As Long = 1) As Collection
    Set GetSelectOptions = New Collection
    FindElement Locator, LocType, index
    Dim result As Object
    Set result = ExecuteScript("var s=arguments[0];var r=[];for(var i=0;i<s.options.length;i++){r.push({value:s.options[i].value,text:s.options[i].text});}return r;", _
                               Array(CreateElementReference(m_CurrentElementId)), 0)
    
    If result.Exists("value") Then
        If TypeName(result("value")) = "Collection" Then
            Dim opt As Object
            Dim i As Long
            For i = 1 To result("value").count
                Set opt = result("value")(i)
                GetSelectOptions.Add opt
            Next i
        End If
    End If
End Function

'===============================================
' Public Methods - Checkbox & Radio
'===============================================
Public Sub Check(ByVal Locator As String, ByVal LocType As LocatorType, _
                 Optional ByVal index As Long = 1)
    If Not IsElementSelected(Locator, LocType, index) Then
        Click Locator, LocType, index, 0
    End If
End Sub

Public Sub Uncheck(ByVal Locator As String, ByVal LocType As LocatorType, _
                   Optional ByVal index As Long = 1)
    If IsElementSelected(Locator, LocType, index) Then
        Click Locator, LocType, index, 0
    End If
End Sub

'===============================================
' Public Methods - Screenshot
'===============================================
Public Sub TakeScreenshot(ByVal FilePath As String)
    Dim result As Object
    Set result = sendRequest("GET", SessionUrl & "/screenshot", Nothing)
    
    If Not result.Exists("value") Then
        RaiseError "XN[Vbg̎擾Ɏs܂"
    End If
    
    Dim base64Data As String: base64Data = result("value")
    Dim bytes() As Byte
    bytes = DecodeBase64(base64Data)
    
    Dim FolderPath As String
    FolderPath = fso.GetParentFolderName(FilePath)
    If FolderPath <> "" And Not fso.FolderExists(FolderPath) Then
        EnsureFolderExists FolderPath
    End If
    
    Dim fileNum As Integer: fileNum = FreeFile
    Open FilePath For Binary Access Write As #fileNum
    Put #fileNum, , bytes
    Close #fileNum
End Sub

Public Sub TakeElementScreenshot(ByVal FilePath As String, ByVal Locator As String, _
                                 ByVal LocType As LocatorType, Optional ByVal index As Long = 1)
    FindElement Locator, LocType, index
    Dim result As Object
    Set result = sendRequest("GET", SessionUrl & "/element/" & m_CurrentElementId & "/screenshot", Nothing)
    
    If Not result.Exists("value") Then
        RaiseError "vf̃XN[Vbg̎擾Ɏs܂"
    End If
    
    Dim base64Data As String: base64Data = result("value")
    Dim bytes() As Byte
    bytes = DecodeBase64(base64Data)
    
    Dim FolderPath As String
    FolderPath = fso.GetParentFolderName(FilePath)
    If FolderPath <> "" And Not fso.FolderExists(FolderPath) Then
        EnsureFolderExists FolderPath
    End If
    
    Dim fileNum As Integer: fileNum = FreeFile
    Open FilePath For Binary Access Write As #fileNum
    Put #fileNum, , bytes
    Close #fileNum
End Sub

'===============================================
' Public Methods - Window Control
'===============================================
Public Function CloseCurrentWindow() As Boolean
    Dim handles As Collection: Set handles = getAllWindowHandles()
    Dim currentHandle As String: currentHandle = getCurrentWindowHandle()
    
    If handles Is Nothing Or currentHandle = "" Then
        CloseCurrentWindow = False
        Exit Function
    End If
    
    sendRequest "DELETE", SessionUrl & "/window", Nothing
    
    If handles.count > 1 Then
        Dim i As Long
        For i = 1 To handles.count
            If handles(i) = currentHandle Then
                Dim nextIndex As Long
                nextIndex = IIf(i = handles.count, 1, i + 1)
                SwitchToWindow handles(nextIndex)
                CloseCurrentWindow = True
                Exit Function
            End If
        Next i
    End If
    CloseCurrentWindow = False
End Function

Public Function SwitchToRightWindow() As Boolean
    SwitchToRightWindow = SwitchWindowByOffset(1)
End Function

Public Function SwitchToLeftWindow() As Boolean
    SwitchToLeftWindow = SwitchWindowByOffset(-1)
End Function

Public Function SwitchToWindowByTitle(ByVal title As String, _
                                      Optional ByVal PartialMatch As Boolean = False) As Boolean
    Dim handles As Collection: Set handles = getAllWindowHandles()
    If handles Is Nothing Then
        SwitchToWindowByTitle = False
        Exit Function
    End If
    
    Dim originalHandle As String: originalHandle = getCurrentWindowHandle()
    Dim h As Variant, pageTitle As String
    
    For Each h In handles
        SwitchToWindow CStr(h)
        pageTitle = GetPageTitle()
        If PartialMatch Then
            If InStr(pageTitle, title) > 0 Then
                SwitchToWindowByTitle = True
                Exit Function
            End If
        Else
            If pageTitle = title Then
                SwitchToWindowByTitle = True
                Exit Function
            End If
        End If
    Next h
    
    SwitchToWindow originalHandle
    SwitchToWindowByTitle = False
End Function

Public Function SwitchToWindowByUrl(ByVal url As String, _
                                    Optional ByVal PartialMatch As Boolean = False) As Boolean
    Dim handles As Collection: Set handles = getAllWindowHandles()
    If handles Is Nothing Then
        SwitchToWindowByUrl = False
        Exit Function
    End If
    
    Dim originalHandle As String: originalHandle = getCurrentWindowHandle()
    Dim h As Variant, pageUrl As String
    
    For Each h In handles
        SwitchToWindow CStr(h)
        pageUrl = GetCurrentUrl()
        If PartialMatch Then
            If InStr(pageUrl, url) > 0 Then
                SwitchToWindowByUrl = True
                Exit Function
            End If
        Else
            If pageUrl = url Then
                SwitchToWindowByUrl = True
                Exit Function
            End If
        End If
    Next h
    
    SwitchToWindow originalHandle
    SwitchToWindowByUrl = False
End Function

Public Function GetWindowCount() As Long
    Dim handles As Collection: Set handles = getAllWindowHandles()
    If handles Is Nothing Then
        GetWindowCount = 0
    Else
        GetWindowCount = handles.count
    End If
End Function

Public Sub NewWindow()
    Dim params As Object: Set params = CreateObject("Scripting.Dictionary")
    params.Add "type", "window"
    sendRequest "POST", SessionUrl & "/window/new", params
    SwitchToRightWindow
End Sub

Public Sub NewTab()
    Dim params As Object: Set params = CreateObject("Scripting.Dictionary")
    params.Add "type", "tab"
    sendRequest "POST", SessionUrl & "/window/new", params
    SwitchToRightWindow
End Sub

Public Sub FullscreenWindow()
    sendRequest "POST", SessionUrl & "/window/fullscreen", CreateObject("Scripting.Dictionary")
End Sub

Public Sub MaximizeWindow()
    sendRequest "POST", SessionUrl & "/window/maximize", CreateObject("Scripting.Dictionary")
End Sub

Public Sub MinimizeWindow()
    sendRequest "POST", SessionUrl & "/window/minimize", CreateObject("Scripting.Dictionary")
End Sub

Public Sub SetWindowPosition(ByVal x As Long, ByVal y As Long)
    Dim rect As Object: Set rect = CreateObject("Scripting.Dictionary")
    rect.Add "x", x
    rect.Add "y", y
    sendRequest "POST", SessionUrl & "/window/rect", rect
End Sub

Public Sub SetWindowSize(ByVal width As Long, ByVal height As Long)
    Dim rect As Object: Set rect = CreateObject("Scripting.Dictionary")
    rect.Add "width", width
    rect.Add "height", height
    sendRequest "POST", SessionUrl & "/window/rect", rect
End Sub

Public Function GetWindowPosition() As Object
    Set GetWindowPosition = CreateObject("Scripting.Dictionary")
    Dim result As Object
    Set result = sendRequest("GET", SessionUrl & "/window/rect", Nothing)
    If result.Exists("value") Then
        GetWindowPosition.Add "x", result("value")("x")
        GetWindowPosition.Add "y", result("value")("y")
    End If
End Function

Public Function GetWindowSize() As Object
    Set GetWindowSize = CreateObject("Scripting.Dictionary")
    Dim result As Object
    Set result = sendRequest("GET", SessionUrl & "/window/rect", Nothing)
    If result.Exists("value") Then
        GetWindowSize.Add "width", result("value")("width")
        GetWindowSize.Add "height", result("value")("height")
    End If
End Function

'===============================================
' Public Methods - Frame/iFrame
'===============================================
Public Sub SwitchToFrame(ByVal Locator As String, ByVal LocType As LocatorType, _
                         Optional ByVal index As Long = 1)
    FindElement Locator, LocType, index
    Dim params As Object: Set params = CreateObject("Scripting.Dictionary")
    params.Add "id", CreateElementReference(m_CurrentElementId)
    sendRequest "POST", SessionUrl & "/frame", params
End Sub

Public Sub SwitchToFrameByIndex(ByVal FrameIndex As Long)
    Dim params As Object: Set params = CreateObject("Scripting.Dictionary")
    params.Add "id", FrameIndex
    sendRequest "POST", SessionUrl & "/frame", params
End Sub

Public Sub SwitchToParentFrame()
    sendRequest "POST", SessionUrl & "/frame/parent", CreateObject("Scripting.Dictionary")
End Sub

Public Sub SwitchToDefaultContent()
    Dim params As Object: Set params = CreateObject("Scripting.Dictionary")
    params.Add "id", Null
    sendRequest "POST", SessionUrl & "/frame", params
End Sub

'===============================================
' Public Methods - Alert
'===============================================
Public Function IsAlertPresent() As Boolean
    On Error Resume Next
    Dim result As Object
    Set result = sendRequest("GET", SessionUrl & "/alert/text", Nothing)
    
    If Err.Number = 0 And result.Exists("value") Then
        If Not IsObject(result("value")) Then
            IsAlertPresent = True
        Else
            IsAlertPresent = False
        End If
    Else
        IsAlertPresent = False
    End If
    On Error GoTo 0
End Function

Public Function GetAlertText() As String
    Dim result As Object
    Set result = sendRequest("GET", SessionUrl & "/alert/text", Nothing)
    If result.Exists("value") Then
        GetAlertText = result("value")
    End If
End Function

Public Sub AcceptAlert()
    sendRequest "POST", SessionUrl & "/alert/accept", CreateObject("Scripting.Dictionary")
End Sub

Public Sub DismissAlert()
    sendRequest "POST", SessionUrl & "/alert/dismiss", CreateObject("Scripting.Dictionary")
End Sub

Public Sub SendAlertText(ByVal text As String)
    Dim params As Object: Set params = CreateObject("Scripting.Dictionary")
    params.Add "text", text
    sendRequest "POST", SessionUrl & "/alert/text", params
End Sub

'===============================================
' Public Methods - Cookies
'===============================================
Public Sub DeleteAllCookies()
    sendRequest "DELETE", SessionUrl & "/cookie", Nothing
End Sub

Public Sub DeleteCookie(ByVal name As String)
    sendRequest "DELETE", SessionUrl & "/cookie/" & name, Nothing
End Sub

Public Function GetCookie(ByVal name As String) As Object
    Set GetCookie = sendRequest("GET", SessionUrl & "/cookie/" & name, Nothing)("value")
End Function

Public Function GetAllCookies() As Collection
    Dim result As Object
    Set result = sendRequest("GET", SessionUrl & "/cookie", Nothing)
    If result.Exists("value") Then
        Set GetAllCookies = result("value")
    Else
        Set GetAllCookies = New Collection
    End If
End Function

Public Sub AddCookie(ByVal name As String, ByVal Value As String, _
                     Optional ByVal path As String = "/", _
                     Optional ByVal domain As String = "", _
                     Optional ByVal secure As Boolean = False, _
                     Optional ByVal HttpOnly As Boolean = False)
    Dim cookie As Object: Set cookie = CreateObject("Scripting.Dictionary")
    cookie.Add "name", name
    cookie.Add "value", Value
    cookie.Add "path", path
    If domain <> "" Then cookie.Add "domain", domain
    cookie.Add "secure", secure
    cookie.Add "httpOnly", HttpOnly
    
    Dim params As Object: Set params = CreateObject("Scripting.Dictionary")
    params.Add "cookie", cookie
    sendRequest "POST", SessionUrl & "/cookie", params
End Sub

'===============================================
' Public Methods - Cleanup
'===============================================

Public Sub DeleteProfile(ByVal ProfileName As String)
    Dim profilePath As String
    profilePath = BasePath & PROFILE_FOLDER & "\" & SanitizeFileName(ProfileName)
    
    If fso.FolderExists(profilePath) Then
        On Error Resume Next
        fso.DeleteFolder profilePath, True
        If Err.Number <> 0 Then
            RaiseError "vt@C̍폜Ɏs܂: " & ProfileName & vbCrLf & Err.Description
        End If
        On Error GoTo 0
    End If
End Sub

Public Sub DeleteAllProfiles()
    Dim profilesPath As String
    profilesPath = BasePath & PROFILE_FOLDER
    
    If fso.FolderExists(profilesPath) Then
        Dim folder As Object
        On Error Resume Next
        For Each folder In fso.GetFolder(profilesPath).SubFolders
            fso.DeleteFolder folder.path, True
        Next folder
        On Error GoTo 0
    End If
End Sub

Public Function GetProfileNames() As Collection
    Set GetProfileNames = New Collection
    Dim profilesPath As String
    profilesPath = BasePath & PROFILE_FOLDER
    
    If fso.FolderExists(profilesPath) Then
        Dim folder As Object
        For Each folder In fso.GetFolder(profilesPath).SubFolders
            GetProfileNames.Add folder.name
        Next folder
    End If
End Function

Public Sub CleanupAll()
    If fso.FolderExists(BasePath) Then
        On Error Resume Next
        fso.DeleteFolder BasePath, True
        If Err.Number <> 0 Then
            RaiseError "N[AbvɎs܂: " & vbCrLf & Err.Description
        End If
        On Error GoTo 0
    End If
End Sub

Public Property Get WorkingFolderPath() As String
    WorkingFolderPath = BasePath
End Property

'===============================================
' Private Methods - Element Finding
'===============================================
Private Sub FindElement(ByVal Locator As String, ByVal LocType As LocatorType, ByVal index As Long)
    m_CurrentElementId = ""
    Dim result As Object
    
    Select Case LocType
        Case ById
            Set result = ExecuteScript("return document.getElementById('" & EscapeJsString(Locator) & "');", , 0)
            If result.Exists("value") Then
                If Not IsNull(result("value")) Then
                    If TypeName(result("value")) = "Dictionary" Then
                        If result("value").Exists(ELEMENT_ID_KEY) Then
                            m_CurrentElementId = result("value")(ELEMENT_ID_KEY)
                        End If
                    End If
                End If
            End If
            
        Case ByName, ByClassName, ByTagName
            Dim jsMethod As String
            Select Case LocType
                Case ByName: jsMethod = "getElementsByName"
                Case ByClassName: jsMethod = "getElementsByClassName"
                Case ByTagName: jsMethod = "getElementsByTagName"
            End Select
            Set result = ExecuteScript("return document." & jsMethod & "('" & EscapeJsString(Locator) & "');", , 0)
            If result.Exists("value") Then
                If TypeName(result("value")) = "Collection" Then
                    If result("value").count >= index Then
                        m_CurrentElementId = result("value")(index)(ELEMENT_ID_KEY)
                    End If
                End If
            End If
            
        Case ByXPath, ByCssSelector, ByLinkText, ByPartialLinkText
            Dim strategy As String
            Select Case LocType
                Case ByXPath: strategy = "xpath"
                Case ByCssSelector: strategy = "css selector"
                Case ByLinkText: strategy = "link text"
                Case ByPartialLinkText: strategy = "partial link text"
            End Select
            
            Dim params As Object
            Set params = CreateObject("Scripting.Dictionary")
            params.Add "using", strategy
            params.Add "value", Locator
            
            On Error Resume Next
            Set result = sendRequest("POST", SessionUrl & "/element", params)
            If Err.Number = 0 Then
                If result.Exists("value") Then
                    If TypeName(result("value")) = "Dictionary" Then
                        If result("value").Exists(ELEMENT_ID_KEY) Then
                            m_CurrentElementId = result("value")(ELEMENT_ID_KEY)
                        End If
                    End If
                End If
            End If
            On Error GoTo 0
    End Select
    
    If m_CurrentElementId = "" Then
        m_LastError = "vf܂ł: " & Locator
    End If
End Sub

Private Function FindElements(ByVal Locator As String, ByVal LocType As LocatorType) As Collection
    Set FindElements = New Collection
    Dim result As Object
    
    Select Case LocType
        Case ById
            Set result = ExecuteScript("var e=document.getElementById('" & EscapeJsString(Locator) & "');return e?[e]:[];", , 0)
            
        Case ByName, ByClassName, ByTagName
            Dim jsMethod As String
            Select Case LocType
                Case ByName: jsMethod = "getElementsByName"
                Case ByClassName: jsMethod = "getElementsByClassName"
                Case ByTagName: jsMethod = "getElementsByTagName"
            End Select
            Set result = ExecuteScript("return Array.from(document." & jsMethod & "('" & EscapeJsString(Locator) & "'));", , 0)
            
        Case ByXPath, ByCssSelector, ByLinkText, ByPartialLinkText
            Dim strategy As String
            Select Case LocType
                Case ByXPath: strategy = "xpath"
                Case ByCssSelector: strategy = "css selector"
                Case ByLinkText: strategy = "link text"
                Case ByPartialLinkText: strategy = "partial link text"
            End Select
            
            Dim params As Object
            Set params = CreateObject("Scripting.Dictionary")
            params.Add "using", strategy
            params.Add "value", Locator
            
            On Error Resume Next
            Set result = sendRequest("POST", SessionUrl & "/elements", params)
            On Error GoTo 0
    End Select
    
    If result Is Nothing Then Exit Function
    If Not result.Exists("value") Then Exit Function
    If TypeName(result("value")) <> "Collection" Then Exit Function
    
    Dim elem As Object
    Dim i As Long
    For i = 1 To result("value").count
        Set elem = CreateObject("Scripting.Dictionary")
        If TypeName(result("value")(i)) = "Dictionary" Then
            elem.Add ELEMENT_ID_KEY, result("value")(i)(ELEMENT_ID_KEY)
            FindElements.Add elem
        End If
    Next i
End Function

Private Function CreateElementReference(ByVal elementId As String) As Object
    Set CreateElementReference = CreateObject("Scripting.Dictionary")
    CreateElementReference.Add ELEMENT_ID_KEY, elementId
End Function

'===============================================
' Private Methods - Window Helpers
'===============================================
Private Function getAllWindowHandles() As Collection
    On Error Resume Next
    Dim result As Object
    Set result = sendRequest("GET", SessionUrl & "/window/handles", Nothing)
    If result.Exists("value") And TypeName(result("value")) = "Collection" Then
        Set getAllWindowHandles = result("value")
    End If
    On Error GoTo 0
End Function

Private Function getCurrentWindowHandle() As String
    On Error Resume Next
    Dim result As Object
    Set result = sendRequest("GET", SessionUrl & "/window", Nothing)
    If result.Exists("value") And Not IsObject(result("value")) Then
        getCurrentWindowHandle = result("value")
    End If
    On Error GoTo 0
End Function

Private Sub SwitchToWindow(ByVal Handle As String)
    Dim params As Object
    Set params = CreateObject("Scripting.Dictionary")
    params.Add "handle", Handle
    sendRequest "POST", SessionUrl & "/window", params
End Sub

Private Function SwitchWindowByOffset(ByVal Offset As Long) As Boolean
    Dim handles As Collection: Set handles = getAllWindowHandles()
    Dim currentHandle As String: currentHandle = getCurrentWindowHandle()
    
    If handles Is Nothing Or currentHandle = "" Or handles.count <= 1 Then
        SwitchWindowByOffset = False
        Exit Function
    End If
    
    Dim i As Long
    For i = 1 To handles.count
        If handles(i) = currentHandle Then
            Dim newIndex As Long
            newIndex = i + Offset
            If newIndex < 1 Then newIndex = handles.count
            If newIndex > handles.count Then newIndex = 1
            SwitchToWindow handles(newIndex)
            SwitchWindowByOffset = True
            Exit Function
        End If
    Next i
    SwitchWindowByOffset = False
End Function

'===============================================
' Private Methods - HTTP Communication
'===============================================
Private Function sendRequest(ByVal method As String, ByVal url As String, ByVal data As Object) As Object
    Set sendRequest = SendRequestInternal(method, url, data, False)
End Function

Private Function SendRequestInternal(ByVal method As String, ByVal url As String, _
                                     ByVal data As Object, ByVal IgnoreErrors As Boolean) As Object
    Dim http As Object
    Set http = CreateObject("MSXML2.ServerXMLHTTP")
    
    http.setTimeouts 5000, 5000, 30000, 60000
    
    On Error Resume Next
    http.Open method, url, False
    If Err.Number <> 0 Then
        If Not IgnoreErrors Then
            RaiseError "HTTPڑG[: " & url & vbCrLf & Err.Description
        End If
        Set SendRequestInternal = CreateObject("Scripting.Dictionary")
        Exit Function
    End If
    On Error GoTo 0
    
    If method = "POST" Or method = "PUT" Then
        http.setRequestHeader "Content-Type", "application/json"
        http.send ToJson(data)
    Else
        http.send
    End If
    
    Dim timeout As Double: timeout = Timer + 60
    Do While http.readyState < 4
        DoEvents
        If Timer > timeout Then
            If Not IgnoreErrors Then
                RaiseError "HTTPNGXg^CAEg܂: " & url
            End If
            Set SendRequestInternal = CreateObject("Scripting.Dictionary")
            Exit Function
        End If
    Loop
    
    If http.status >= 400 And Not IgnoreErrors Then
        Dim errDetail As String
        errDetail = "HTTP Error " & http.status & ": " & http.statusText & vbCrLf & _
                    "URL: " & url & vbCrLf & _
                    "Response: " & Left(http.responseText, 500)
        m_LastError = errDetail
    End If
    
    If Len(Trim(http.responseText)) = 0 Then
        Set SendRequestInternal = CreateObject("Scripting.Dictionary")
    Else
        Set SendRequestInternal = ParseJson(http.responseText)
    End If
End Function

'===============================================
' Private Methods - JSON (Minimal Implementation)
'===============================================
Private Function ParseJson(ByVal JsonStr As String) As Object
    If Len(Trim(JsonStr)) = 0 Then
        Set ParseJson = CreateObject("Scripting.Dictionary")
        Exit Function
    End If
    
    Dim idx As Long: idx = 1
    JsonStr = Replace(Replace(Replace(JsonStr, vbCr, ""), vbLf, ""), vbTab, "")
    SkipWhitespace JsonStr, idx
    
    If idx > Len(JsonStr) Then
        Set ParseJson = CreateObject("Scripting.Dictionary")
        Exit Function
    End If
    
    Select Case Mid(JsonStr, idx, 1)
        Case "{": Set ParseJson = ParseObject(JsonStr, idx)
        Case "[": Set ParseJson = ParseArray(JsonStr, idx)
        Case Else
            Set ParseJson = CreateObject("Scripting.Dictionary")
    End Select
End Function

Private Function ParseObject(ByRef s As String, ByRef idx As Long) As Object
    Dim result As Object
    Set result = CreateObject("Scripting.Dictionary")
    idx = idx + 1
    
    Dim Key As String
    Dim val As Variant
    Dim c As String
    
    Do
        SkipWhitespace s, idx
        If idx > Len(s) Then Exit Do
        
        c = Mid(s, idx, 1)
        
        If c = "}" Then
            idx = idx + 1
            Set ParseObject = result
            Exit Function
        End If
        
        If c = "," Then
            idx = idx + 1
            SkipWhitespace s, idx
            If idx > Len(s) Then Exit Do
            c = Mid(s, idx, 1)
        End If
        
        If c = "}" Then
            idx = idx + 1
            Set ParseObject = result
            Exit Function
        End If
        
        If c <> """" And c <> "'" Then
            Exit Do
        End If
        
        Key = ParseString(s, idx)
        SkipWhitespace s, idx
        
        If idx > Len(s) Then Exit Do
        If Mid(s, idx, 1) = ":" Then idx = idx + 1
        SkipWhitespace s, idx
        
        If idx > Len(s) Then Exit Do
        
        Dim isObj As Boolean: isObj = False
        c = Mid(s, idx, 1)
        If c = "{" Or c = "[" Then isObj = True
        
        If result.Exists(Key) Then result.Remove Key
        
        If isObj Then
            Dim objVal As Object
            If c = "{" Then
                Set objVal = ParseObject(s, idx)
            Else
                Set objVal = ParseArray(s, idx)
            End If
            result.Add Key, objVal
        Else
            val = ParseValue(s, idx)
            If IsNull(val) Then
                result.Add Key, Null
            ElseIf IsEmpty(val) Then
                result.Add Key, ""
            Else
                result.Add Key, val
            End If
        End If
    Loop
    
    Set ParseObject = result
End Function
Private Function ParseArray(ByRef s As String, ByRef idx As Long) As Collection
    Set ParseArray = New Collection
    idx = idx + 1
    
    Dim val As Variant
    Dim isObj As Boolean
    
    Do
        SkipWhitespace s, idx
        If idx > Len(s) Then Exit Do
        
        Dim c As String: c = Mid(s, idx, 1)
        
        If c = "]" Then
            idx = idx + 1
            Exit Function
        End If
        
        If c = "," Then
            idx = idx + 1
            SkipWhitespace s, idx
            If idx > Len(s) Then Exit Do
            c = Mid(s, idx, 1)
        End If
        
        If c = "]" Then
            idx = idx + 1
            Exit Function
        End If
        
        isObj = (c = "{" Or c = "[")
        
        If isObj Then
            Dim objVal As Object
            Set objVal = ParseValue(s, idx)
            ParseArray.Add objVal
        Else
            val = ParseValue(s, idx)
            ParseArray.Add val
        End If
    Loop
End Function

Private Function ParseValue(ByRef s As String, ByRef idx As Long) As Variant
    SkipWhitespace s, idx
    If idx > Len(s) Then
        ParseValue = Null
        Exit Function
    End If
    
    Dim c As String: c = Mid(s, idx, 1)
    
    Select Case c
        Case "{"
            Set ParseValue = ParseObject(s, idx)
        Case "["
            Set ParseValue = ParseArray(s, idx)
        Case """", "'"
            ParseValue = ParseString(s, idx)
        Case Else
            If idx + 3 <= Len(s) And LCase(Mid(s, idx, 4)) = "true" Then
                ParseValue = True: idx = idx + 4
            ElseIf idx + 4 <= Len(s) And LCase(Mid(s, idx, 5)) = "false" Then
                ParseValue = False: idx = idx + 5
            ElseIf idx + 3 <= Len(s) And LCase(Mid(s, idx, 4)) = "null" Then
                ParseValue = Null: idx = idx + 4
            ElseIf InStr("+-0123456789", c) > 0 Then
                ParseValue = ParseNumber(s, idx)
            Else
                ' \Ȃ - XLbvĎ
                ParseValue = Null
                idx = idx + 1
            End If
    End Select
End Function
Private Function ParseString(ByRef s As String, ByRef idx As Long) As String
    If idx > Len(s) Then
        ParseString = ""
        Exit Function
    End If
    
    Dim quote As String: quote = Mid(s, idx, 1)
    
    If quote <> """" And quote <> "'" Then
        ParseString = ""
        Exit Function
    End If
    
    idx = idx + 1
    
    Dim result As String, c As String
    Dim startIdx As Long: startIdx = idx
    
    Do While idx <= Len(s)
        c = Mid(s, idx, 1)
        If c = "\" Then
            idx = idx + 1
            If idx > Len(s) Then Exit Do
            c = Mid(s, idx, 1)
            Select Case c
                Case """", "\", "/", "'": result = result & c
                Case "b": result = result & vbBack
                Case "f": result = result & vbFormFeed
                Case "n": result = result & vbLf
                Case "r": result = result & vbCr
                Case "t": result = result & vbTab
                Case "u"
                    If idx + 4 <= Len(s) Then
                        Dim hexCode As String
                        hexCode = Mid(s, idx + 1, 4)
                        On Error Resume Next
                        result = result & ChrW(CLng("&H" & hexCode))
                        If Err.Number <> 0 Then
                            result = result & "?"
                            Err.Clear
                        End If
                        On Error GoTo 0
                        idx = idx + 4
                    End If
                Case Else
                    result = result & c
            End Select
        ElseIf c = quote Then
            idx = idx + 1
            ParseString = result
            Exit Function
        Else
            result = result & c
        End If
        idx = idx + 1
    Loop
    
    ParseString = result
End Function
Private Function ParseNumber(ByRef s As String, ByRef idx As Long) As Variant
    Dim numStr As String
    Dim c As String
    
    Do While idx <= Len(s)
        c = Mid(s, idx, 1)
        If InStr("+-0123456789.eE", c) > 0 Then
            numStr = numStr & c
            idx = idx + 1
        Else
            Exit Do
        End If
    Loop
    
    If numStr = "" Or numStr = "-" Or numStr = "+" Then
        ParseNumber = 0
    Else
        ParseNumber = val(numStr)
    End If
End Function

Private Sub SkipWhitespace(ByRef s As String, ByRef idx As Long)
    Do While idx <= Len(s)
        Select Case Mid(s, idx, 1)
            Case " ", vbCr, vbLf, vbTab
                idx = idx + 1
            Case Else
                Exit Do
        End Select
    Loop
End Sub


Private Function ToJson(ByVal Obj As Object) As String
    If Obj Is Nothing Then ToJson = "{}": Exit Function
    
    Dim result As String, Key As Variant, isFirst As Boolean
    result = "{"
    isFirst = True
    
    For Each Key In Obj.Keys
        If Not isFirst Then result = result & ","
        isFirst = False
        result = result & """" & Key & """:"
        
        If IsObject(Obj(Key)) Then
            If TypeName(Obj(Key)) = "Dictionary" Then
                result = result & ToJson(Obj(Key))
            ElseIf TypeName(Obj(Key)) = "Collection" Then
                result = result & CollectionToJson(Obj(Key))
            Else
                result = result & "null"
            End If
        ElseIf IsArray(Obj(Key)) Then
            result = result & ArrayToJson(Obj(Key))
        ElseIf IsNull(Obj(Key)) Then
            result = result & "null"
        ElseIf VarType(Obj(Key)) = vbString Then
            result = result & """" & EscapeJsonString(CStr(Obj(Key))) & """"
        ElseIf VarType(Obj(Key)) = vbBoolean Then
            result = result & IIf(Obj(Key), "true", "false")
        Else
            result = result & Obj(Key)
        End If
    Next Key
    
    ToJson = result & "}"
End Function

Private Function ArrayToJson(ByVal arr As Variant) As String
    Dim result As String, i As Long, isFirst As Boolean
    result = "["
    isFirst = True
    
    If IsArray(arr) Then
        For i = LBound(arr) To UBound(arr)
            If Not isFirst Then result = result & ","
            isFirst = False
            If IsObject(arr(i)) Then
                If TypeName(arr(i)) = "Dictionary" Then
                    result = result & ToJson(arr(i))
                Else
                    result = result & "null"
                End If
            ElseIf VarType(arr(i)) = vbString Then
                result = result & """" & EscapeJsonString(CStr(arr(i))) & """"
            ElseIf VarType(arr(i)) = vbBoolean Then
                result = result & IIf(arr(i), "true", "false")
            ElseIf IsNull(arr(i)) Then
                result = result & "null"
            Else
                result = result & arr(i)
            End If
        Next i
    End If
    
    ArrayToJson = result & "]"
End Function

Private Function CollectionToJson(ByVal col As Collection) As String
    Dim result As String, isFirst As Boolean
    result = "["
    isFirst = True
    
    Dim item As Variant
    Dim i As Long
    For i = 1 To col.count
        If Not isFirst Then result = result & ","
        isFirst = False
        If IsObject(col(i)) Then
            If TypeName(col(i)) = "Dictionary" Then
                result = result & ToJson(col(i))
            Else
                result = result & "null"
            End If
        ElseIf VarType(col(i)) = vbString Then
            result = result & """" & EscapeJsonString(CStr(col(i))) & """"
        Else
            result = result & col(i)
        End If
    Next i
    
    CollectionToJson = result & "]"
End Function

Private Function EscapeJsonString(ByVal s As String) As String
    s = Replace(s, "\", "\\")
    s = Replace(s, """", "\""")
    s = Replace(s, vbCr, "\r")
    s = Replace(s, vbLf, "\n")
    s = Replace(s, vbTab, "\t")
    EscapeJsonString = s
End Function

Private Function EscapeJsString(ByVal s As String) As String
    s = Replace(s, "\", "\\")
    s = Replace(s, "'", "\'")
    s = Replace(s, """", "\""")
    s = Replace(s, vbCr, "\r")
    s = Replace(s, vbLf, "\n")
    EscapeJsString = s
End Function

'===============================================
' Private Methods - Setup & Configuration
'===============================================
Private Function DetectChromeVersion() As String
    Dim chromePath As String
    chromePath = FindChromePath()
    If chromePath <> "" Then
        DetectChromeVersion = fso.GetFileVersion(chromePath)
    End If
End Function

Private Function FindChromePath() As String
    Dim paths(2) As String
    paths(0) = Environ("ProgramFiles(x86)") & "\Google\Chrome\Application\chrome.exe"
    paths(1) = Environ("ProgramW6432") & "\Google\Chrome\Application\chrome.exe"
    paths(2) = Environ("ProgramFiles") & "\Google\Chrome\Application\chrome.exe"
    
    Dim i As Long
    For i = 0 To 2
        If fso.FileExists(paths(i)) Then
            FindChromePath = paths(i)
            Exit Function
        End If
    Next i
End Function

Private Function GetMatchingDriverVersion(ByVal BrowserVer As String) As String
    Dim http As Object: Set http = CreateObject("MSXML2.XMLHTTP")
    Dim parts() As String: parts = Split(BrowserVer, ".")
    Dim majorVer As String: majorVer = Join(Array(parts(0), parts(1), parts(2)), ".")
    
    Dim url As String
    url = "https://chromedriver.storage.googleapis.com/LATEST_RELEASE_" & majorVer
    
    On Error Resume Next
    http.Open "GET", url, False: http.send
    If http.status = 200 Then
        GetMatchingDriverVersion = http.responseText
        Exit Function
    End If
    On Error GoTo 0
    
    url = "https://googlechromelabs.github.io/chrome-for-testing/latest-patch-versions-per-build.json"
    On Error Resume Next
    http.Open "GET", url, False: http.send
    If http.status = 200 Then
        Dim pos As Long, endPos As Long
        pos = InStr(http.responseText, majorVer)
        If pos > 0 Then
            pos = InStr(pos, http.responseText, """version""") + 11
            endPos = InStr(pos, http.responseText, """")
            GetMatchingDriverVersion = Mid(http.responseText, pos, endPos - pos)
        End If
    End If
    On Error GoTo 0
    
    If GetMatchingDriverVersion = "" Then
        RaiseError "ChromeDriver̃o[W擾ł܂łB" & vbCrLf & _
                   "Chrome o[W: " & BrowserVer
    End If
End Function

Private Sub EnsureDriverExists()
    If fso.FileExists(driverPath) Then Exit Sub
    
    Dim zipPath As String
    zipPath = DownloadDriver(m_DriverVersion)
    
    If zipPath = "" Then
        RaiseError "ChromeDriver̃_E[hɎs܂B" & vbCrLf & _
                   "hCo[o[W: " & m_DriverVersion
    End If
    
    Dim unzipPath As String
    unzipPath = Replace(zipPath, ".zip", "")
    EnsureFolderExists unzipPath
    
    If UnzipFile(zipPath, unzipPath) Then
        Dim exePath As String
        exePath = FindExeInFolder(unzipPath)
        If exePath <> "" Then
            Name exePath As driverPath
            On Error Resume Next
            Kill zipPath
            fso.DeleteFolder unzipPath
            On Error GoTo 0
        End If
    Else
        RaiseError "ChromeDriver̉𓀂Ɏs܂B"
    End If
End Sub

Private Function DownloadDriver(ByVal DriverVer As String) As String
    Dim savePath As String, downloadUrl As String
    Dim is64Bit As Boolean: is64Bit = InStr(wsh.Environment("Process")("PROCESSOR_ARCHITECTURE"), "64") > 0
    
    Dim downloadFolder As String
    downloadFolder = Environ("USERPROFILE") & "\Downloads"
    
    If Not fso.FolderExists(downloadFolder) Then
        downloadFolder = BasePath
        EnsureFolderExists downloadFolder
    End If
    
    If is64Bit Then
        savePath = downloadFolder & "\chromedriver-win64_" & DriverVer & ".zip"
    Else
        savePath = downloadFolder & "\chromedriver-win32_" & DriverVer & ".zip"
    End If
    
    If val(Split(DriverVer, ".")(0)) < 115 Then
        downloadUrl = "https://chromedriver.storage.googleapis.com/" & DriverVer & "/chromedriver_win32.zip"
    ElseIf is64Bit Then
        downloadUrl = "https://storage.googleapis.com/chrome-for-testing-public/" & DriverVer & "/win64/chromedriver-win64.zip"
    Else
        downloadUrl = "https://storage.googleapis.com/chrome-for-testing-public/" & DriverVer & "/win32/chromedriver-win32.zip"
    End If
    
    DeleteUrlCacheEntry downloadUrl
    If URLDownloadToFile(0, downloadUrl, savePath, 0, 0) = 0 Then
        DownloadDriver = savePath
    End If
End Function

Private Function UnzipFile(ByVal zipPath As String, ByVal DestPath As String) As Boolean
    On Error Resume Next
    UnzipFile = False
    
    Dim cmd As String
    cmd = "powershell -ExecutionPolicy Bypass -NoProfile -Command ""Expand-Archive -Path '" & zipPath & "' -DestinationPath '" & DestPath & "' -Force"""
    
    Dim exitCode As Long
    exitCode = wsh.Run(cmd, 0, True)
    
    If exitCode = 0 Then
        If fso.GetFolder(DestPath).Files.count > 0 Or fso.GetFolder(DestPath).SubFolders.count > 0 Then
            UnzipFile = True
            Exit Function
        End If
    End If
    
    Err.Clear
    Dim shell As Object: Set shell = CreateObject("Shell.Application")
    Dim zipFolder As Object: Set zipFolder = shell.Namespace(zipPath)
    Dim destFolder As Object: Set destFolder = shell.Namespace(DestPath)
    
    If Not zipFolder Is Nothing And Not destFolder Is Nothing Then
        destFolder.CopyHere zipFolder.Items, &H14  ' &H14 = ㏑mFȂ + i\
        
        Dim waitCount As Long
        Do
            Wait 500
            waitCount = waitCount + 1
            If fso.GetFolder(DestPath).Files.count > 0 Or fso.GetFolder(DestPath).SubFolders.count > 0 Then
                UnzipFile = True
                Exit Function
            End If
        Loop While waitCount < 20
    End If
    
    On Error GoTo 0
End Function

Private Function FindExeInFolder(ByVal FolderPath As String) As String
    Dim f As Object
    For Each f In fso.GetFolder(FolderPath).Files
        If LCase(fso.GetExtensionName(f.name)) = "exe" Then
            FindExeInFolder = f.path
            Exit Function
        End If
    Next f
    
    For Each f In fso.GetFolder(FolderPath).SubFolders
        FindExeInFolder = FindExeInFolder(f.path)
        If FindExeInFolder <> "" Then Exit Function
    Next f
End Function

Private Sub CreateBrowserShortcut()
    If fso.FileExists(BrowserShortcutPath) Then
        If Not m_Headless Then Exit Sub
        fso.DeleteFile BrowserShortcutPath
    End If
    
    Dim chromePath As String: chromePath = FindChromePath()
    Dim chromeFolder As String: chromeFolder = fso.GetParentFolderName(chromePath)
    Dim args As String
    args = " --remote-debugging-port=" & m_BrowserPort & _
           " --user-data-dir=""" & BasePath & PROFILE_FOLDER & "\" & m_ProfileName & """"
    
    If m_Headless Then
        args = args & " --headless=new --disable-gpu --window-size=1920,1080"
    End If
    
    Dim sc As Object
    Set sc = wsh.CreateShortcut(BrowserShortcutPath)
    sc.TargetPath = chromePath
    sc.WorkingDirectory = chromeFolder
    sc.Arguments = args
    sc.WindowStyle = 1
    sc.IconLocation = chromePath & ",4"
    sc.Save
    
    If Not m_Headless Then
        Dim userShortcut As String
        userShortcut = ThisWorkbook.path & "\" & BrowserShortcutName
        If Not fso.FileExists(userShortcut) Then
            Set sc = wsh.CreateShortcut(userShortcut)
            sc.TargetPath = chromePath
            sc.WorkingDirectory = chromeFolder
            sc.Arguments = args
            sc.WindowStyle = 1
            sc.IconLocation = chromePath & ",4"
            sc.Save
        End If
    End If
End Sub

Private Function IsBrowserRunning() As Boolean
    On Error Resume Next
    Dim http As Object
    Set http = CreateObject("MSXML2.ServerXMLHTTP")
    http.setTimeouts 500, 500, 500, 500
    http.Open "GET", "http://127.0.0.1:" & m_BrowserPort & "/json/version", False
    http.send
    
    IsBrowserRunning = (http.status = 200)
    On Error GoTo 0
End Function

Private Sub OpenBrowserIfNotRunning()
    If Not IsBrowserRunning Then
        Dim cmd As String
        cmd = """" & GetChromeExePath() & """" & _
              " --remote-debugging-port=" & m_BrowserPort & _
              " --user-data-dir=""" & BasePath & PROFILE_FOLDER & "\" & m_ProfileName & """" & _
              " --no-first-run" & _
              " --no-default-browser-check" & _
              " --disable-popup-blocking"
        
        If m_Headless Then
            cmd = cmd & " --headless=new --disable-gpu --window-size=1920,1080"
        End If
        
        wsh.Run cmd, 1, False
    End If
End Sub
Private Function ConnectToBrowser() As Boolean
    On Error GoTo Failed
    
    Dim params As Object: Set params = CreateObject("Scripting.Dictionary")
    Dim capabilities As Object: Set capabilities = CreateObject("Scripting.Dictionary")
    Dim chromeOptions As Object: Set chromeOptions = CreateObject("Scripting.Dictionary")
    Dim alwaysMatch As Object: Set alwaysMatch = CreateObject("Scripting.Dictionary")
    
    chromeOptions.Add "debuggerAddress", "127.0.0.1:" & m_BrowserPort
    alwaysMatch.Add "browserName", "chrome"
    alwaysMatch.Add "goog:chromeOptions", chromeOptions
    capabilities.Add "alwaysMatch", alwaysMatch
    params.Add "capabilities", capabilities
    params.Add "desiredCapabilities", CreateObject("Scripting.Dictionary")
    
    Dim http As Object
    Set http = CreateObject("MSXML2.ServerXMLHTTP")
    http.setTimeouts 5000, 5000, 30000, 60000
    http.Open "POST", DriverBaseUrl & "session", False
    http.setRequestHeader "Content-Type", "application/json"
    http.send ToJson(params)
    
    If http.status >= 400 Then GoTo Failed
    
    Dim result As Object
    Set result = ParseJson(http.responseText)
    
    If result.Exists("value") Then
        If TypeName(result("value")) = "Dictionary" Then
            If result("value").Exists("sessionId") Then
                m_SessionId = result("value")("sessionId")
                ConnectToBrowser = True
                Exit Function
            End If
        End If
    End If
    
Failed:
    ConnectToBrowser = False
End Function
'===============================================
' Private Methods - Utilities
'===============================================
Private Function IsValidVersionFormat(ByVal Ver As String) As Boolean
    On Error GoTo Invalid
    Dim parts() As String: parts = Split(Ver, ".")
    If UBound(parts) <> 3 Then GoTo Invalid
    
    Dim i As Long
    For i = 0 To 3
        If Not IsNumeric(parts(i)) Then GoTo Invalid
    Next i
    IsValidVersionFormat = True
    Exit Function
Invalid:
    IsValidVersionFormat = False
End Function

Private Function SanitizeFileName(ByVal name As String) As String
    Dim Invalid As Variant, repl As Variant
    Invalid = Array("\", "/", ":", "*", "?", """", "<", ">", "|")
    repl = Array("", "^", "F", "", "H", "", "", "", "b")
    
    Dim i As Long
    For i = 0 To UBound(Invalid)
        name = Replace(name, Invalid(i), repl(i))
    Next i
    SanitizeFileName = name
End Function

Private Function StringToCharArray(ByVal text As String) As String()
    Dim arr() As String
    If Len(text) = 0 Then
        ReDim arr(0)
        arr(0) = ""
        StringToCharArray = arr
        Exit Function
    End If
    
    ReDim arr(0 To Len(text) - 1)
    Dim i As Long
    For i = 1 To Len(text)
        arr(i - 1) = Mid(text, i, 1)
    Next i
    StringToCharArray = arr
End Function

Private Sub EnsureFolderExists(ByVal FolderPath As String)
    Dim parts() As String, currentPath As String, i As Long
    parts = Split(FolderPath, "\")
    
    For i = LBound(parts) To UBound(parts)
        currentPath = currentPath & parts(i)
        If Not fso.FolderExists(currentPath) And currentPath <> "" Then
            On Error Resume Next
            fso.CreateFolder currentPath
            On Error GoTo 0
        End If
        currentPath = currentPath & "\"
    Next i
End Sub

Private Function DecodeBase64(ByVal base64String As String) As Byte()
    Dim xml As Object: Set xml = CreateObject("MSXML2.DOMDocument")
    Dim node As Object: Set node = xml.createElement("b64")
    node.DataType = "bin.base64"
    node.text = base64String
    DecodeBase64 = node.nodeTypedValue
End Function

Private Sub RaiseError(ByVal Message As String)
    m_LastError = Message
    Err.Raise vbObjectError + 1001, "LesserScraping", Message
End Sub

Private Function GetChromeExePath() As String
    Dim paths(2) As String
    paths(0) = Environ("Programfiles(x86)") & "\Google\Chrome\Application\chrome.exe"
    paths(1) = Environ("ProgramW6432") & "\Google\Chrome\Application\chrome.exe"
    paths(2) = Environ("Programfiles") & "\Google\Chrome\Application\chrome.exe"
    
    Dim i As Long
    For i = 0 To 2
        If fso.FileExists(paths(i)) Then
            GetChromeExePath = paths(i)
            Exit Function
        End If
    Next i
    GetChromeExePath = ""
End Function

'===============================================
' Private Properties - Cached Objects
'===============================================
Private Property Get fso() As Object
    If m_FSO Is Nothing Then Set m_FSO = CreateObject("Scripting.FileSystemObject")
    Set fso = m_FSO
End Property

Private Property Get wsh() As Object
    If m_WSH Is Nothing Then Set m_WSH = CreateObject("WScript.Shell")
    Set wsh = m_WSH
End Property

'===============================================
' Private Properties - Paths & URLs
'===============================================
Private Property Get BasePath() As String
    BasePath = Environ("LOCALAPPDATA") & "\" & BASE_FOLDER
End Property

Private Property Get driverPath() As String
    driverPath = BasePath & DRIVER_FOLDER & "\chromedriver_" & m_DriverVersion & ".exe"
End Property

Private Property Get BrowserShortcutName() As String
    BrowserShortcutName = "GoogleChromei" & m_ProfileName & "F" & m_BrowserPort & "j.lnk"
End Property

Private Property Get BrowserShortcutPath() As String
    BrowserShortcutPath = BasePath & BROWSER_FOLDER & "\" & BrowserShortcutName
End Property

Private Property Get DriverBaseUrl() As String
    DriverBaseUrl = "http://localhost:" & m_DriverPort & "/"
End Property

Private Property Get SessionUrl() As String
    SessionUrl = DriverBaseUrl & "session/" & m_SessionId
End Property

